/*
 * scroll_cells2.c -- scroll a bitmap of cells around in a canvas.
 * This is a simplified version of scroll_cells.c graphically. That
 * is, it does not display icons, just rows and columns of cells.
 * The difference with this version is that it attempts to accommodate
 * resize events not addressed in the scroll_cells.c.
 * This new function is at the end of the file.
 */
#include <stdio.h>
#include <X11/X.h>
#include <X11/Xlib.h>   /* Using Xlib graphics */
#include <xview/xview.h>
#include <xview/canvas.h>
#include <xview/scrollbar.h>
#include <xview/xv_xrect.h>

#define CELL_WIDTH              64
#define CELL_HEIGHT             64
#define CELLS_PER_HOR_PAGE      5 /* when paging w/scrollbar */
#define CELLS_PER_VER_PAGE      5 /* when paging w/scrollbar */
#define CELLS_PER_ROW           16
#define CELLS_PER_COL           16

Pixmap          cell_map;       /* pixmap copied onto canvas window */
Scrollbar       horiz_scrollbar;
Scrollbar       vert_scrollbar;
GC              gc;             /* General usage GC */

main(argc, argv)
int argc;
char *argv[];
{
    Frame       frame;
    Canvas      canvas;
    void        repaint_proc(), resize_proc();

    /* Initialize, create frame and canvas... */
    xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, NULL);

    frame = xv_create(XV_NULL, FRAME,
        FRAME_LABEL,            argv[0],
        FRAME_SHOW_FOOTER,      TRUE,
        NULL);

    canvas = xv_create(frame, CANVAS,
        /* make subwindow the size of a "page" */
        XV_WIDTH,               CELL_WIDTH * CELLS_PER_HOR_PAGE,
        XV_HEIGHT,              CELL_HEIGHT * CELLS_PER_VER_PAGE,
        /* canvas is same size as window */
        CANVAS_WIDTH,           CELL_WIDTH * CELLS_PER_HOR_PAGE,
        CANVAS_HEIGHT,          CELL_HEIGHT * CELLS_PER_VER_PAGE,
        /* don't retain window -- we'll repaint it all the time */
        CANVAS_RETAINED,        FALSE,
        /* We're using Xlib graphics calls in repaint_proc() */
        CANVAS_X_PAINT_WINDOW,  TRUE,
        CANVAS_REPAINT_PROC,    repaint_proc,
        CANVAS_RESIZE_PROC,     resize_proc,
        OPENWIN_AUTO_CLEAR,     FALSE,
        NULL);

    /*
     * Create scrollbars attached to the canvas. When user clicks
     * on cable, page by the page size (PAGE_LENGTH). Scrolling
     * should move cell by cell, not by one pixel (PIXELS_PER_UNIT).
     */
    vert_scrollbar = xv_create(canvas, SCROLLBAR,
        SCROLLBAR_DIRECTION,            SCROLLBAR_VERTICAL,
        SCROLLBAR_PIXELS_PER_UNIT,      CELL_HEIGHT,
        NULL);
    horiz_scrollbar = xv_create(canvas, SCROLLBAR,
        SCROLLBAR_DIRECTION,            SCROLLBAR_HORIZONTAL,
        SCROLLBAR_PIXELS_PER_UNIT,      CELL_WIDTH,
        NULL);

    /*
     * create pixmap and draw cells into it.  This portion of the
     * program could use XCopyArea to render real bitmaps whose sizes
     * do not exceed whatever CELL_WIDTH and CELL_HEIGHT are defined
     * to be.  The cell_map will be copied into the window via
     * XCopyPlane in the repaint procedure.
     */
    {
        short           x, y, pt = 0;
        XPoint          points[256];
        XGCValues       gcvalues;
        Display *dpy = (Display *)xv_get(canvas, XV_DISPLAY);

        cell_map = XCreatePixmap(dpy, DefaultRootWindow(dpy),
            CELLS_PER_ROW * CELL_WIDTH + 1,
            CELLS_PER_COL * CELL_HEIGHT + 1,
            1); /* We only need a 1-bit deep pixmap */

        /* Create the gc for the cell_map -- since it is 1-bit deep,
         * use 0 and 1 for fg/bg values.  Also, limit number of
         * events generated by setting geraphics exposures to False.
         */
        gcvalues.graphics_exposures = False;
	gcvalues.foreground = WhitePixel (dpy, DefaultScreen (dpy) );
	gc = XCreateGC (dpy, cell_map,
       	     		GCForeground | GCGraphicsExposures, &gcvalues);
	XFillRectangle (dpy, cell_map, gc, 0, 0, CELLS_PER_ROW * CELL_WIDTH + 1,
			CELLS_PER_COL * CELL_HEIGHT + 1);
	XSetForeground (dpy, gc, 1L);
	XSetBackground (dpy, gc, 0L);


        /* dot every other pixel */
        for (x = 0; x <= CELL_WIDTH * CELLS_PER_ROW; x += 2)
            for (y = 0; y <= CELL_HEIGHT * CELLS_PER_COL; y += 2) {
                if (x % CELL_WIDTH != 0 && y % CELL_HEIGHT != 0)
                    continue;
                points[pt].x = x, points[pt].y = y;
                if (++pt == sizeof points / sizeof points[0]) {
                    XDrawPoints(dpy, cell_map, gc,
                        points, pt, CoordModeOrigin);
                    pt = 0;
                }
            }
        if (pt != sizeof points) /* flush out the remaining points */
            XDrawPoints(dpy, cell_map, gc,
                        points, pt, CoordModeOrigin);
        /* label each cell indicating the its coordinates */
        for (x = 0; x < CELLS_PER_ROW; x++)
            for (y = 0; y < CELLS_PER_COL; y++) {
                char buf[8];
                sprintf(buf, "%d,%d", x+1, y+1);
                XDrawString(dpy, cell_map, gc,
                    x * CELL_WIDTH + 5, y * CELL_HEIGHT + 25,
                    buf, strlen(buf));
            }
        /* we're now done with the cell_map, so free gc and
         * create a new one based on the window that will use it.
         */
        XFreeGC(dpy, gc);
        gcvalues.background = WhitePixel(dpy, DefaultScreen(dpy));
        gcvalues.foreground = BlackPixel(dpy, DefaultScreen(dpy));
        gcvalues.plane_mask = 1L;
        gc = XCreateGC(dpy, DefaultRootWindow(dpy),
            GCForeground|GCBackground|GCGraphicsExposures, &gcvalues);
    }

    /* shrink frame to minimal size and start notifier */
    window_fit(frame);
    xv_main_loop(frame);
}

/*
 * The repaint procedure is called whenever repainting is needed in
 * a paint window.  Since the canvas is not retained, this routine
 * is going to be called only when the user scrolls to a portion of the 
 * canvas that is currently not visible.  The canvas will 
 * handle repainting the portion of the canvas that was in view 
 * and has scrolled onto another viewable portion of the window.  
 * The xrects parameter will cover the new areas that were not in 
 * view before and have just scrolled into view.  If the window 
 * resizes or if the window is exposed by other windows disappearing 
 * or cycling thru the window tree, then the number of xrects will 
 * be more than one and we'll have to copy the new areas one by one.  
 * Clipping isn't necessary since the areas to be rendered are set 
 * by the xrects value.
 */
void
repaint_proc(canvas, paint_window, dpy, win, xrects)
Canvas          canvas;
Xv_Window       paint_window;
Display         *dpy;
Window          win;
Xv_xrectlist    *xrects;
{
    int x, y;

    x = (int)xv_get(horiz_scrollbar, SCROLLBAR_VIEW_START);
    y = (int)xv_get(vert_scrollbar, SCROLLBAR_VIEW_START);

    for (xrects->count--; xrects->count >= 0; xrects->count--) {
        printf("top-left cell = %d, %d -- %d,%d %d,%d\n", x+1, y+1,
            xrects->rect_array[xrects->count].x,
            xrects->rect_array[xrects->count].y,
            xrects->rect_array[xrects->count].width,
            xrects->rect_array[xrects->count].height);

        XCopyPlane(dpy, cell_map, win, gc,
            x * CELL_WIDTH,
            y * CELL_HEIGHT,
            xv_get(paint_window, XV_WIDTH),
            xv_get(paint_window, XV_HEIGHT),
            0, 0, 1L);
    }
}

/*
 * If the application is resized, then we may wish to reset the
 * paging and viewing parameters for the scrollbars.
 */
void
resize_proc(canvas, new_width, new_height)
Canvas canvas;
int new_width, new_height;
{
    int page_w = (int)(new_width/CELL_WIDTH);
    int page_h = (int)(new_height/CELL_HEIGHT);

    if (!vert_scrollbar || !horiz_scrollbar)
        return;

    printf("new width/height in cells: w = %d, h = %d\n",
        page_w, page_h);

    xv_set(horiz_scrollbar,
        SCROLLBAR_OBJECT_LENGTH,        CELLS_PER_ROW,
        SCROLLBAR_PAGE_LENGTH,          page_w,
        SCROLLBAR_VIEW_LENGTH,          page_w,
        NULL);
    xv_set(vert_scrollbar,
        SCROLLBAR_OBJECT_LENGTH,        CELLS_PER_COL,
        SCROLLBAR_PAGE_LENGTH,          page_h,
        SCROLLBAR_VIEW_LENGTH,          page_h,
        NULL);
}
