
/*  @(#)dsfuns.c 1.3 02/02/27
 *
 *  External functions needed by calctool from the libdeskset, libguide
 *  and libguidexv libraries.
 *
 *  Copyright (c) 1987-2002  Sun Microsystems, Inc.  All Rights Reserved.
 *
 *  Sun considers its source code as an unpublished, proprietary
 *  trade secret, and it is available only under strict license
 *  provisions.  This copyright notice is placed here only to protect
 *  Sun in the event the source is deemed a published work.  Dissassembly,
 *  decompilation, or other means of reducing the object code to human
 *  readable form is prohibited by the license agreement under which
 *  this code is provided to the user or company in posesion of this copy.
 *
 *  RESTRICTED RIGHTS LEGEND:  Use, duplication, or disclosure by the
 *  Government is subject to restrictions as set forth in subparagraph
 *  (c)(1)(ii) of the Rights in Technical Data and Computer Software
 *  clause at DFARS 52.227-7013 and in similar clauses in the FAR and
 *  NASA FAR Supplement.
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/param.h>
#include <sys/stat.h>
#include "dsdefs.h"
#include <stdio.h>
#include <sys/types.h>
#include <strings.h>
#include <ctype.h>
#include <pwd.h>
#include <dirent.h>
#include <netdb.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <xview/xview.h>
#include <xview/panel.h>

#define GETHOSTNAME  (void) gethostname
#define SPRINTF      (void) sprintf
#define STRCPY       (void) strcpy
#define STRNCPY      (void) strncpy

#define DS_MAX_STR   120          /* Maximum length for character strings. */

static int ds_position_popup_rect(Rect *, Frame, enum ds_location_op);
static int ds_force_popup_on_screen(int *, int *, Frame);
static int ds_get_screen_size(Frame, int *, int *);


/* Function definitions from .../libdeskset/ds_xlib.c */

/*  Function:     ds_add_help()
 *
 *  Purpose:      Add a help string to an XView item.
 *
 *  Parameters:   item       Xv_opaque
 *                           An opaque handle to the XView item.
 *
 *                str        The help string to be associated with this item.
 *
 *  Returns:      XV_ERROR   if unable to assign help text.
 *                XV_OK      on successful completion.
 */

Xv_opaque
ds_add_help(Xv_opaque item, char *str)
{
    return(xv_set(item, XV_HELP_DATA, str, 0));
}


/*  Function:     ds_beep()
 *
 *  Purpose:      Ring the bell (at base volume).
 *
 *  Parameters:   display    connection to the X server.
 *                           (returned from XOpenDisplay).
 *
 *  Returns:      None.
 */

void
ds_beep(Display *display)
{
    XBell(display, 0);
}


/*  Function:     ds_get_font()
 * 
 *  Purpose:      Get an Xlib handle to a font.
 * 
 *  Parameters:   display    connection to the X server.
 *                           (returned from XOpenDisplay).
 *
 *                fontname   name of font to load. If this is NULL, or the
 *                           load fails, then ds_get_font() tries to load the
 *                           lucidatypewriter font. If this fails then the
 *                           fixed font is tried.
 *
 *                size       If fontname was NULL, then this indicates the
 *                           size (scale) of the lucidatypewriter font to load.
 * 
 *  Returns:      A pointer to an XFontStruct, or NULL if a font can't be
 *                loaded.
 */

XFontStruct *
ds_get_font(Display *display, char *fontname, int size)
{
    char fname[DS_MAX_STR];    /* Used to construct the required font name. */
    XFontStruct *font;

    SPRINTF(fname, 
            "-b&h-lucidatypewriter-medium-r-*-*-*-%1d0-*-*-*-*-iso8859-1",
            size);
    if (fontname == NULL || !(font = XLoadQueryFont(display, fontname))) {
        if (!(font = XLoadQueryFont(display, fname))) {
            font = XLoadQueryFont(display, "fixed");
        }
    }

    return(font);
}


/*  Function:     ds_get_frame_size()
 *
 *  Purpose:      Return the bounding box of a window.
 *
 *  Parameters:   window     XView handle to the window.
 *
 *  Returns:      Parameters:  x          left most value.
 *                             y          top most value.
 *                             width      width of bounding box.
 *                             height     height of bounding box.
 *
 *                Value:       A pointer to the Rect structure for this window
 *                             or NULL for an invalid window.
 */

Rect *
ds_get_frame_size(Xv_window window, int *x, int *y, int *width, int *height)
{
    Rect *r;

    r = (Rect *) xv_get(window, WIN_RECT);
    if (r != NULL) {
        *x      = r->r_left;
        *y      = r->r_top;
        *width  = r->r_width;
        *height = r->r_height;
    }

    return(r);
}


/*  Function:     ds_get_geometry_size()
 *
 *  Purpose:      Return width and height values from an X geometry string.
 *
 *  Parameters:   str          X geometry string.
 *
 *  Returns:      Parameters:  width      width   (-1 if invalid).
 *                             height     height  (-1 if invalid).
 *
 *                Value:       None.
 */

void
ds_get_geometry_size(char *str, int *width, int *height)
{
    int reply, x, y;
    unsigned int w, h;

    *width = *height = -1;
    reply = XParseGeometry(str, &x, &y, &w, &h);
    if (reply & WidthValue) {
        *width  = w;
    }
    if (reply & HeightValue) {
        *height = h;
    }
}


/*  Function:     ds_get_resource()
 *
 *  Purpose:      Get an X resource from the server.
 *
 *  Parameters:   rDB          X resources database.
 *
 *                appname      application name.
 *
 *                resource     X resource string to search for.
 *
 *  Returns:      resource string, or NULL if not found.
 *
 *  Note:         The first character of the appname and resource strings may
 *                be modified.
 */

char *
ds_get_resource(XrmDatabase rDB, char *appname, char *resource)          
{
    char cstr[DS_MAX_STR], nstr[DS_MAX_STR], *str_type[20];
    XrmValue value;

    if (isupper((int) appname[0])) {
        appname[0] = tolower((int) appname[0]);
    }
    SPRINTF(nstr, "deskset.%s.%s", appname, resource);

    if (islower((int) resource[0])) {
        resource[0] = toupper((int) resource[0]);
    }
    if (islower((int) appname[0])) {
        appname[0] = toupper((int) appname[0]);
    }
    SPRINTF(cstr, "Deskset.%s.%s", appname, resource);

    if (XrmGetResource(rDB, nstr, cstr, str_type, &value) == False) {
        return((char *) NULL);
    } else {
        return(value.addr);
    }
}


/*  Function:     ds_get_strwidth()
 *
 *  Purpose:      Get the length in pixels of a text string, given the font
 *                it would be written in.
 *
 *  Parameters:   font         font XFontStruct pointer.
 *
 *                str          string
 *
 *  Returns:      length in pixels of the text string.
 */

int
ds_get_strwidth(XFontStruct *font, char *str)
{
    return(XTextWidth(font, str, strlen(str)));
}


/*  Function:     ds_load_deskset_defs()
 *
 *  Purpose:      Load the Deskset resources database from the
 *                DESKSETDEFAULTS environment variable (if set), or
 *                from $HOME/.desksetdefaults.
 *
 *  Parameters:   None.
 *
 *  Returns:      X resources database.
 */

XrmDatabase
ds_load_deskset_defs()
{
    XrmDatabase rDB = NULL;
    char name[MAXPATHLEN], *ptr;

    if ((ptr = getenv("DESKSETDEFAULTS")) == NULL) {
        if ((ptr = getenv("HOME")) != NULL) {
            SPRINTF(name, "%s/.desksetdefaults", ptr);
            rDB = XrmGetFileDatabase(name);
        }
    } else {
        rDB = XrmGetFileDatabase(ptr);
    }

    return(rDB);
}


/*  Function:     ds_load_resources()
 *
 *  Purpose:      Get the resource databases. These are looked for in the
 *                following ways:
 *
 *                Classname file in the app-defaults directory. In this case,
 *                Classname is Deskset.
 *
 *                Classname file in the directory specified by the
 *                XUSERFILESEARCHPATH or XAPPLRESDIR environment variable.
 *
 *                Property set using xrdb, accessible through the
 *                XResourceManagerString macro or, if that is empty, the
 *                ~/.Xdefaults file.
 *
 *                XENVIRONMENT environment variable or, if not set,
 *                .Xdefaults-hostname file.
 *
 *                DESKSETDEFAULTS environment variable or, if not set, the
 *                ~/.desksetdefaults file
 *
 *  Parameters:   display    connection to the X server.
 *                           (returned from XOpenDisplay).
 *
 *  Returns:      X combined resources database.
 */

XrmDatabase
ds_load_resources(Display *display)
{
    XrmDatabase db, rDB;
    char *home, name[MAXPATHLEN], *owhome, *ptr;
    int len;

    rDB  = NULL;
    home = getenv("HOME");
    XrmInitialize();
    if ((owhome = getenv("OPENWINHOME")) != NULL) {
        SPRINTF(name, "%s/lib/app-defaults/Deskset", owhome);

/* Get applications defaults file, if any. */

        db = XrmGetFileDatabase(name);
        XrmMergeDatabases(db, &rDB);
    }

/* Merge server defaults, created by xrdb. If nor defined, use ~/.Xdefaults. */

    if (XResourceManagerString(display) != NULL) {
        db = XrmGetStringDatabase(XResourceManagerString(display));
        XrmMergeDatabases(db, &rDB);
    } else if (home != NULL) { 
        SPRINTF(name, "%s/.Xdefaults", home);
        db = XrmGetFileDatabase(name);
        XrmMergeDatabases(db, &rDB);
    }

/*  Open XENVIRONMENT file or, if not defined, the .Xdefaults, and merge
 *  into existing database.
 */

    if ((ptr = getenv("XENVIRONMENT")) == NULL) {
        if (home != NULL) {
            SPRINTF(name, "%s/.Xdefaults-", home);
            len = strlen(name);
            GETHOSTNAME(name+len, 1024-len);
            db = XrmGetFileDatabase(name);
            XrmMergeDatabases(db, &rDB);
        }
    } else {
        db = XrmGetFileDatabase(ptr);
        XrmMergeDatabases(db, &rDB);
    }

/*  Finally merge in Deskset defaults via DESKSETDEFAULTS or, if not
 *  defined, the ~/.desksetdefaults file.
 */

    if ((ptr = getenv("DESKSETDEFAULTS")) == NULL) {
        if (home != NULL) {
            SPRINTF(name, "%s/.desksetdefaults", home);
            db = XrmGetFileDatabase(name);
            XrmMergeDatabases(db, &rDB);
        }
    } else {
        db = XrmGetFileDatabase(ptr);
        XrmMergeDatabases(db, &rDB);
    }

    return(rDB);
}


/*  Function:     ds_put_resource()
 *
 *  Purpose:      Adds an X resource string (name and value) to a resources
 *                database.
 *
 *  Parameters:   rDB          X resources database.
 *
 *                appname      application name.
 *
 *                rstr         X resource string name.
 *
 *                rval         X resource string value.
 *
 *  Returns:      None.
 *
 *  Note:         The first character of the appname and resource strings may
 *                be modified.
 */

void
ds_put_resource(XrmDatabase *rDB, char *appname, char *rstr, char *rval)
{
    char resource[DS_MAX_STR];

    if (isupper((int) appname[0])) {
        appname[0] = tolower((int) appname[0]);
    }
    SPRINTF(resource, "deskset.%s.%s", appname, rstr);
    XrmPutStringResource(rDB, resource, rval);
}


/*  Function:     ds_save_cmdline()
 *
 *  Purpose:      Save away the application command line options.
 *
 *  Parameters:   frame        Applications main XView base frame.
 *
 *                argc         Number of command line options.
 *
 *                argv         An array of command line options.
 *
 *  Returns:      XV_ERROR   if unable to save command line.
 *                XV_OK      on successful completion.
 */

Xv_opaque
ds_save_cmdline(Frame frame, int argc, char *argv[])
{
    return(xv_set(frame, FRAME_WM_COMMAND_ARGC_ARGV, argc, argv, 0));
}


/*  Function:     ds_save_resources()
 *
 *  Purpose:      Save away the resources database to the file given by the
 *                DESKSETDEFAULTS environment variable (if set), or
 *                to $HOME/.desksetdefaults.
 *
 *  Parameters:   rDB        X resources database to save.
 *
 *  Returns:      XV_ERROR   if cannot access resource database to write.
 *                XV_OK      on successful completion.
 */

Xv_opaque
ds_save_resources(XrmDatabase rDB)
{
    char *home, *filename;
    struct stat statbuf;

    if ((filename = getenv("DESKSETDEFAULTS")) == NULL) {
        if ((home = getenv("HOME")) == NULL) {
            return(XV_ERROR);
        }
        filename = (char*) calloc(1, strlen(home) + 18);
        SPRINTF(filename, "%s/.desksetdefaults", home);
    }

    /* If file exists but user does not have access. */
    if (stat(filename, &statbuf) != -1 && access(filename, W_OK) != 0) { 
        free(filename);
        return(XV_ERROR);
    }

    /* If file does not exist this call will create it. */
    XrmPutFileDatabase(rDB, filename);
    free(filename);

    return(XV_OK);
}


/*  Function:     ds_set_frame_size()
 *
 *  Purpose:      Sets the bounding box of a window.
 *
 *  Parameters:   window     XView handle to the window.
 *
 *                x          left most value.
 *                y          top most value.
 *                width      width of bounding box.
 *                height     height of bounding box.
 *
 *  Returns:      XV_ERROR   if the operation was successful.
 *                XV_OK      on successful completion.
 */

Xv_opaque
ds_set_frame_size(Xv_window window, int x, int y, int width, int height)
{
    Rect r;

    r.r_left   = x;
    r.r_top    = y;
    r.r_width  = width;
    r.r_height = height;

    return(xv_set(window, WIN_RECT, &r, 0));
}

/* ========================================================================= */

/* Function definitions from .../libdeskset/ds_popup.c */

/*
 * Function:    ds_position_popup
 *
 * Description: Position a popup relative to the parent frame
 *              making sure it doesn't go off of the screen.
 *
 * Parameters:  base           Popup's parent frame
 *              popup          Popup frame
 *              location_op    Where you would like the popup to
 *                             appear.  Location_op may be any
 *                             the following:
 *
 * DS_POPUP_LEFT      Place the popup to the left of base with the tops flush
 * DS_POPUP_RIGHT     Place the popup to the right of base with the tops flush
 * DS_POPUP_ABOVE     Place the popup above base with the left edges flush
 * DS_POPUP_BELOW     Place the popup below base with the left edges flush
 * DS_POPUP_LOR       Place the popup either to the left or right of base
 *                    depending on which side has the most space.
 * DS_POPUP_AOF       Place the popup either above or below base
 *                    depending on which side has the most space.
 * DS_POPUP_CENTERED  Center popup within baseframe
 *
 * Returns:        0    Could not get screen size
 *                 1    All is well
 */

int
ds_position_popup(Frame base, Frame popup, enum ds_location_op location_op)
{
    Rect base_rect;

    frame_get_rect(base, &base_rect);

    return(ds_position_popup_rect(&base_rect, popup, location_op));
}


/*
 *  Function:    ds_position_popup_rect
 *
 *  Description: Position a popup relative to the parent frame
 *               making sure it doesn't go off of the screen.
 *
 *               We added this rect interface to support the "move"
 *               tooltalk message, since the app will not have
 *               the baseframe, just its rect.
 *    
 *
 *  Parameters:  base_rect_p    Pointer to parent's frame rect
 *               popup          popup to position
 *               location_op    Where you would like the popup to
 *                              appear.  Location_op may be any
 *                              the following:
 *
 *  DS_POPUP_LEFT      Place the popup to the left of base with the tops flush
 *  DS_POPUP_RIGHT     Place the popup to the right of base with the tops flush
 *  DS_POPUP_ABOVE     Place the popup above base with the left edges flush
 *  DS_POPUP_BELOW     Place the popup below base with the left edges flush
 *  DS_POPUP_LOR       Place the popup either to the left or right of base
 *                     depending on which side has the most space.
 *  DS_POPUP_AOF       Place the popup either above or below base
 *                     depending on which side has the most space.
 *  DS_POPUP_CENTERED  Center popup within baseframe
 *
 *  Returns:        0    Could not get screen size
 *                  1    All is well
 */

static int
ds_position_popup_rect(Rect *base_rect_p, Frame popup, 
                       enum ds_location_op location_op)
{
    Rect popup_rect;
    int screen_width, screen_height;
    int base_x, base_y, popup_x, popup_y, base_width, base_height;
    int popup_width, popup_height; 

    frame_get_rect(popup, &popup_rect);

    ds_get_screen_size(popup, &screen_width, &screen_height);

    base_x = base_rect_p->r_left;
    base_y = base_rect_p->r_top;
    base_width = base_rect_p->r_width;
    base_height = base_rect_p->r_height;
    popup_width = popup_rect.r_width;
    popup_height = popup_rect.r_height;

    if (location_op == DS_POPUP_LOR) {
        if (base_x >= screen_width - base_width - base_x) {
            location_op = DS_POPUP_LEFT;
        } else {
            location_op = DS_POPUP_RIGHT;
        }
    } else if (location_op == DS_POPUP_AOB) {
        if (base_y > screen_height - base_height - base_y) {
            location_op = DS_POPUP_ABOVE;
        } else {
            location_op = DS_POPUP_BELOW;
        }
    }

    switch (location_op) {
        case DS_POPUP_RIGHT:
            popup_x = base_x + base_width;
            popup_y = base_y;
            break;

        case DS_POPUP_LEFT:
            popup_x = base_x - popup_width;
            popup_y = base_y;
            break;

        case DS_POPUP_ABOVE:
            popup_x = base_x;
            popup_y = base_y - popup_height;
            break;

        case DS_POPUP_BELOW:
            popup_x = base_x;
            popup_y = base_y + base_height;
            break;

        case DS_POPUP_CENTERED:
        default:
            popup_x = base_x + (base_width - popup_width) / 2;
            popup_y = base_y + (base_height - popup_height) / 2;
            break;
    }

    ds_force_popup_on_screen(&popup_x, &popup_y, popup);

    return(1);
}

/*
 *  Function:    ds_force_popup_on_screen
 *
 *  Description: Make sure that the specified frame appears entirely
 *               on the screen.
 *
 *               You specify the x and y where you would like the
 *               popup to appear.  If this location would cause any
 *               portion of the popup to appear off of the screen
 *               then the routine makes the minimum adjustments 
 *               necessary to move it onto the screen.
 *
 *  NOTE:        The following coordinates must be specified
 *               relative to the screen origin *not* the
 *               parent frame! (as in frame_rects as
 *               opposed to XV_X, XV_Y);
 *
 *  Parameters: popup_x_p    Pointer to x location where you would
 *                           like the popup to appear.  If the popup
 *                           is moved this is updated to reflect
 *                           the new position.
 *              popup_y_p    Pointer to y location where you would
 *                           like the popup to appear.  If the popup
 *                           is moved this is updated to reflect
 *                           the new position.
 *              popup        Popup`s frame
 *
 *  Returns:  TRUE    The popup was moved
 *            FALSE    The popup was not moved
 *
 */

/*ARGSUSED*/
static int
ds_force_popup_on_screen(int *popup_x_p, int *popup_y_p, Frame popup)
{
    Rect popup_rect;
    int popup_x, popup_y, screen_width, screen_height;
    int popup_width, popup_height, n, rcode;

    popup_x = *popup_x_p;
    popup_y = *popup_y_p;

    /* Get the screen size */
    ds_get_screen_size(popup, &screen_width, &screen_height);

    frame_get_rect(popup, &popup_rect);
    popup_width = popup_rect.r_width;
    popup_height = popup_rect.r_height;

    /* Make sure frame does not go off side of screen */
    n = popup_x + popup_width;
    if (n > screen_width) {
        popup_x -= (n - screen_width);
    } else if (popup_x < 0) {
        popup_x = 0;
    }

    /* Make sure frame doen't go off top or bottom */
    n = popup_y + popup_height;
    if (n > screen_height) {
        popup_y -= n - screen_height;
    } else if (popup_y < 0) {
        popup_y = 0;
    }

    /* Set location and return */
    popup_rect.r_left = popup_x;
    popup_rect.r_top = popup_y;
    frame_set_rect(popup, &popup_rect);

    if (popup_x != *popup_x_p || popup_y != *popup_y_p) {
        rcode = TRUE;
    } else {
        rcode = FALSE;
    }
    *popup_x_p = popup_x;
    *popup_y_p = popup_y;

    return(rcode);
}

/*
 *  Function:       ds_get_screen_size
 *
 *  Description:    Get the width and height of the screen in pixels
 *
 *  Parameters:     width_p     Pointer to an integer to place width
 *                  height_p    Pointer to an integer to place height
 *
 *  Returns:        1    All is well
 *
 */

static int
ds_get_screen_size(Frame frame, int *width_p, int *height_p)
{
    Rect *scr_size;

    scr_size = (Rect *)xv_get(frame, WIN_SCREEN_RECT);
    *width_p = scr_size->r_width;
    *height_p = scr_size->r_height;

    return(1);
}

/*
 *
 *  Function:       ds_center_items
 *
 *  Description:    Center items on a given row
 *
 *  Parameters:     panel      Panel containing items
 *                  row        Row to place items on. -1 to use y
 *                             location of first item.
 *                  Null terminated list of items to center.
 *
 *           ie:    ds_center_items(panel, 2, button1, button2, button3, 0);
 *
 *  Note that the panel must have its width set before calling this function.
 *
 *  Returns:        x location of leftmost item
 *
 */

/* VARARGS1 */
int
ds_center_items(Panel panel, int row, ...)
{
    va_list args;
    int y;                /* y location */
    int x;                /* x location */
    int width;            /* Total width of items + gaps */
    int x_gap;            /* Gap between items */
    Rect *rect;           /* Item rects */
    int rcode;
    Panel_item    item;

    x_gap = (int) xv_get(panel, PANEL_ITEM_X_GAP);

    if (row < 0) {
        y = -1;
    } else {
        y = xv_row(panel, row);
    }

    /* Sum up total width of all items and gaps. */

    width = 0;

    va_start(args, row);

    while ((item = (Panel_item ) va_arg(args, Panel_item *)) != 0) {
        if (y < 0) {
            y = (int) xv_get(item, XV_Y);
        }
        rect = (Rect *) xv_get(item, PANEL_ITEM_RECT);
        width += rect->r_width + x_gap;
    }
    va_end(args);

    width -= x_gap;

    /* Set x location of leftmost item. */

    x = ((int) xv_get(panel, XV_WIDTH) - width) / 2;

    if (x < 0) {
        x = 0;
    }

    rcode = x;

    /* Position items. */

    va_start(args, row);

    while ((item = (Panel_item ) va_arg(args, Panel_item *)) != 0) {
        xv_set(item, XV_X, x, XV_Y, y, 0);
        rect = (Rect *)xv_get(item, PANEL_ITEM_RECT);
        x += rect->r_width + x_gap;
    }
    va_end(args);

    return(rcode);
}

/* ========================================================================= */

/* Function definition from .../libdeskset/ds_pathname.c */

/*
 * Function:    ds_expand_pathname
 *
 * Description: expand ~'s and environment variables in a path.
 *        
 * This routine was stolen from filemgr. I've made a couple of small 
 * changes to better handle "~user" at the start of a path.
 *
 * Parameters:  path - unexpanded path
 *              buf  - returned expanded path.
 *                     Caller is responsible allocating enough space.
 *
 * Returns:     Nothing
 *
 */

void
ds_expand_pathname(char *path, char *buf)
{
    char *home, *p, *b_p, *e_p;
    char *save_p;            /* Point in path before env var */
    char env[255];            /* Environment variable expansion */
    struct passwd *pw;        /* Password file entry */

    p = path;
    if (*p == '~') {
        p++;
        if (*p && *p != '/') {
            /* Somebody else's home directory? */
            if ((b_p = strchr(p, '/'))) {
                *b_p = '\0';
            }
            if ((pw = getpwnam(p))) {
                STRCPY(buf, pw->pw_dir);
            } else {
                *buf = '~';
                STRNCPY(buf + 1, p, MAXPATHLEN - 2);
                if (strlen(p) > MAXPATHLEN - 3) {
                    buf[MAXPATHLEN - 1] = '\0';
                }
            }

            if (b_p)  {
                *b_p = '/';
                p = b_p;
            } else {
                return;
            }
        } else if ((home = (char *)getenv("HOME")) != NULL) {
            STRNCPY(buf, home, MAXPATHLEN - 1); /* fix for 4153829 */
            if (strlen(home) > MAXPATHLEN - 2) {
                buf[MAXPATHLEN - 1] = '\0';
            }
        }
        b_p = buf + strlen(buf);
    } else {
        b_p = buf;
    }

    while (*p)
        if (*p == '$') {
            /* Expand environment variable */
            save_p = p;
            e_p = env;
            p++;
            while ((isalnum((int) *p) || *p == '_') && e_p - env < 255) {
                *e_p++ = *p++;
            }
            *e_p = '\0';
            if ((e_p = (char *) getenv(env))) {
                while ((*e_p) && b_p - buf < MAXPATHLEN - 1) {
                    *b_p++ = *e_p++;
                }
            } else {
                p = save_p;
                if (b_p - buf < MAXPATHLEN - 1) {
                    *b_p++ = *p++;
                } else {
                    *p = '\0';
                }
            }
        } else {
            if (b_p - buf < MAXPATHLEN - 1) {
                *b_p++ = *p++;
            } else {
                *p = '\0';
            }
        }
 
    *b_p = '\0';
}


/* ========================================================================= */

/* Function definition from .../libdeskset/ds_hostname.c */

#define TRUE  1
#define FALSE 0

static char client_hostname[MAXHOSTNAMELEN + 4];
static int ds_hostname_set = 0;


char *
ds_hostname(Display *dpy)
{
    char hostname[MAXHOSTNAMELEN];
    char *display = DisplayString(dpy);
    char *scanner = display;

    if (ds_hostname_set) {
        return(client_hostname);
    }

    GETHOSTNAME(hostname, MAXHOSTNAMELEN);

    while (*scanner) {
        scanner++;
    }

    while (*scanner != ':') {
        scanner--;
    }

    *scanner = '\0';

    if (strcmp(display, hostname) && 
        strcmp(display, "localhost") &&
        strcmp(display, "unix") &&
        strcmp(display, "")) {
        SPRINTF(client_hostname, " [%s] ", hostname);
    } else {
        STRCPY(client_hostname, "");
    }

    *scanner = ':';

    ds_hostname_set = TRUE;

    return(client_hostname);
}

/* ========================================================================= */
