/*
 * $Id: cddb.c,v 1.3 1999/10/27 05:48:05 dirk Exp $
 *
 * This file is part of WorkMan, the civilized CD player program
 * (c) 1991-1997 by Steven Grimm (original author)
 * (c) by Dirk F"orsterling (current 'author' = maintainer)
 * The maintainer can be contacted by his e-mail address:
 * milliByte@DeathsDoor.com 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * establish connection to cddb server and get data
 * socket stuff gotten from gnu port of the finger command
 *
 * provided by Sven Oliver Moll
 *
 */

static char cddb_id[] = "$Id: cddb.c,v 1.3 1999/10/27 05:48:05 dirk Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/signal.h>
#include <ctype.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include "config.h"
#include "struct.h"
#include "cdinfo.h"

#ifdef DEBUG
#define Trace(x)  fprintf x
#else
#define Trace(x)
#endif

#define PROGRAM "WorkMan"
#define VERSION "1.3.4"

struct wm_cddb cddb;

int cur_cddb_protocol;
char *cur_cddb_server;
char *cur_cddb_mail_adress;
char *cur_cddb_path_to_cgi;
char *cur_cddb_proxy_server;

int Socket;
FILE *Connection;

/*
 *
 */
void
cddb_cur2struct(void)
{
	cddb.protocol = cur_cddb_protocol;
	strcpy(cddb.cddb_server,  cur_cddb_server);
	strcpy(cddb.mail_adress,  cur_cddb_mail_adress);
	strcpy(cddb.path_to_cgi,  cur_cddb_path_to_cgi);
	strcpy(cddb.proxy_server, cur_cddb_proxy_server);
} /* cddb_cur2struct() */
void
cddb_struct2cur(void)
{
	cur_cddb_protocol = cddb.protocol;
	cur_cddb_server =       (cddb.cddb_server);
	cur_cddb_mail_adress =  (cddb.mail_adress);
	cur_cddb_path_to_cgi =  (cddb.path_to_cgi);
	cur_cddb_proxy_server = (cddb.proxy_server);
} /* cddb_struct2cur() */

/*
 * Subroutine from cddb_discid
 */
int
cddb_sum(int n)
{
	char	buf[12],
		*p;
	int	ret = 0;

	/* For backward compatibility this algorithm must not change */
	sprintf(buf, "%lu", (unsigned long)n);
	for (p = buf; *p != '\0'; p++)
	  ret += (*p - '0');

	return (ret);
} /* cddb_sum() */


/*
 * Calculate the discid of a CD according to cddb
 */
unsigned long
cddb_discid(void)
{
	int	i,
		t,
		n = 0;

	/* For backward compatibility this algorithm must not change */
	for (i = 0; i < thiscd.ntracks; i++) {
	    
		n += cddb_sum(thiscd.trk[i].start / 75);
	/* 
	 * Just for demonstration (See below)
	 * 
	 *	t += (thiscd.trk[i+1].start / 75) -
	 *	     (thiscd.trk[i  ].start / 75);
	 */	    
	}

	/*
	 * Mathematics can be fun. Example: How to reduce a full loop to
	 * a simple statement. The discid algorhythm is so half-hearted
	 * developed that it doesn't even use the full 32bit range.
	 * But it seems to be always this way: The bad standards will be
	 * accepted, the good ones turned down.
	 */

	t = (thiscd.trk[thiscd.ntracks].start-thiscd.trk[0].start)/75;
	return ((n % 0xff) << 24 | t << 8 | thiscd.ntracks);
} /* cddb_discid() */

/* Split string at delim */
char *string_split(char *line, char delim)
{
	char *p1;
	char *strchr();

	p1 = strchr(line, (int)delim);
	if (p1) {
			*p1 = 0;
			return ++p1;
		}
	return (NULL);
} /* string_split() */

/*
 * Generate the hello string according to the cddb protocol
 * delimiter is either ' ' (cddbp) or '+' (http)
 */
void
string_makehello(char *line,char delim)
{
	char mail[84], *host;
	int len;
	
	strcpy(mail,cddb.mail_adress);
	host = string_split(mail,'@');
	if (!host) {
	    if (getlogin())
	        strncpy(mail, getlogin(), 83);
	    else
	        strcpy(mail, "nobody");
	    len = strlen(mail);
	    host = mail + len + 1;
	    gethostname(host, 83-len);
	    Trace((stderr, "default email %s@%s\n", mail, host)); 
	}
	sprintf(line,"%shello%c%s%c%s%c%s%c%s",
		delim == ' ' ? "cddb " : "&",
		delim == ' ' ? ' ' : '=',
		mail,delim,
		host,delim,
		PROGRAM,delim,
		VERSION);
} /* string_makehello() */

/*
 * Open the TCP connection to the cddb/proxy server
 */
int
connect_open(void)
{
	char host[200], *prt;
	struct hostent *hp;
       	struct sockaddr_in soc_in;
       	int port = 0;
 
	Trace((stderr, "opening connection to cddb server\n"));
 
	/* in case of failure */
	Connection = NULL;
	
	if(cddb.protocol == 3) /* http proxy */
	    strcpy(host, cddb.proxy_server);
	else
	    strcpy(host, cddb.cddb_server);
 
	Trace((stderr, "host %s\n", host));
 
	prt = string_split(host,':'); 
	if (prt)
	    port = atoi(prt);
	
	/* Defaults */
	if(!port)
	  {
	    if (cddb.protocol == 1)
		port = 888;
	    else
		port = 80;
	  }
	if (cddb.path_to_cgi[0] == 0)
	    sprintf(cddb.path_to_cgi, "/~cddb/cddb.cgi");
 
	Trace((stderr, "%s : %d\n",host,port));
	
	hp = gethostbyname(host);
	if (!hp)
	{
		static struct hostent def;
		static struct in_addr defaddr;
		static char *alist[1];
		static char namebuf[128];
		int inet_addr();
		
		Trace((stderr, "gethostbyname failed: %s\n", host));
 
		defaddr.s_addr = inet_addr(host);
		if (defaddr.s_addr == -1) 
		{
			Trace((stderr, "unknown host: %s\n", host));
			return (-1);
		}
		strcpy(namebuf, host);
		def.h_name = namebuf;
		def.h_addr_list = alist, def.h_addr = (char *)&defaddr;
		def.h_length = sizeof (struct in_addr);
		def.h_addrtype = AF_INET;
		def.h_aliases = 0;
		hp = &def;
	}
	soc_in.sin_family = hp->h_addrtype;
	bcopy(hp->h_addr, (char *)&soc_in.sin_addr, hp->h_length);
	soc_in.sin_port = htons(port);
	Socket = socket(hp->h_addrtype, SOCK_STREAM, 0);
	if (Socket < 0) 
	{ 
#ifdef DEBUG
		perror("socket");
#endif
		return (-1);
	}
	fflush(stdout);
	if (connect(Socket, (struct sockaddr *)&soc_in, sizeof (soc_in)) < 0) 
	{   
#ifdef DEBUG
		perror("connect");
#endif
		close(Socket);
		return (-1);
	}
	
	Connection = fdopen(Socket, "r");
	return (0);
} /* connect_open() */


/*
 * Close the connection
 */
void
connect_close(void)
{	   
	(void)fclose(Connection);
	close(Socket);
} /* connect_close() */

/*
 * Get a line from the connection with CR and LF stripped
 */
void
connect_getline(char *line)
{
	char c;
    
	while ((c = getc(Connection)) != '\n')
	{
		*line = c;
		if ((c != '\r') && (c != (char)0xff))
		  line++;
	}
	*line=0;
} /* connect_getline() */

/*
 * Read the CD data from the server and place them into the cd struct
       
 */
void
connect_read_entry(void)
{
	char type;
	int trknr;
	
	char *t,*t2,tempbuf[2000];
	
	while(strcmp(tempbuf,"."))
	{
		connect_getline(tempbuf);
		
		t = string_split(tempbuf,'=');
		if (t != NULL)
		{
			type=tempbuf[0];
			
			if(strncmp("TITLE",tempbuf+1,5))
			  continue;
			
       /* FIXME timeout, return empty string (not null) */
    
			if(type == 'D')
			{
				/*
				 * Annahme: "Interpret / Titel" ist falsch.
				 * Daher: NULL-String erwarten.
				 */
				t2=string_split(t,'/');
				if(t2 == NULL)
					t2 = t;
				if(*t2 == ' ')
				  t2++;
				strcpy(cd->cdname,t2);
				
				for(t2=t;*t2;t2++)
				{
					if((*t2 == ' ') && (*(t2+1) == 0))
					  *t2=0;
				}
				strcpy(cd->artist,t);
			}
			else if(type == 'T')
			{
				trknr=atoi(tempbuf+6);
				strmcpy(&cd->trk[trknr].songname,t);
			}
		}
	}
 } /* connect_read_entry() */

/*
 * Send a command to the server using cddbp
 */
void
cddbp_send(char *line)
{
	write(Socket, line, strlen(line));
	write(Socket, "\n", 1);
} /* cddbp_send() */

/*
 * Send the "read from cddb" command to the server using cddbp
 */
void
cddbp_read(char *category, unsigned int id)
{
	char tempbuf[84];
	sprintf(tempbuf, "cddb read %s %08x", category, id);
	cddbp_send(tempbuf);
} /* cddbp_read() */

/*
 * Send a command to the server using http
 */
void
http_send(char* line)
{
	char tempbuf[2000];
	
	write(Socket, "GET ", 4);
	Trace((stderr, "GET "));
	if(cddb.protocol == 3)
	{
		write(Socket, "http://", 7);
		write(Socket, cddb.cddb_server, strlen(cddb.cddb_server));
		Trace((stderr, "http://%s",cddb.cddb_server));
	}
	write(Socket, cddb.path_to_cgi, strlen(cddb.path_to_cgi));
	write(Socket, "?cmd=" ,5);
	write(Socket, line, strlen(line));
	Trace((stderr, "%s?cmd=%s",cddb.path_to_cgi,line));
	string_makehello(tempbuf,'+');
	write(Socket, tempbuf, strlen(tempbuf));
	Trace((stderr, "%s",tempbuf));
	write(Socket, "&proto=1 HTTP/1.0\n\n", 19);
	Trace((stderr, "&proto=1 HTTP/1.0\n"));
	do
	  connect_getline(tempbuf);
	while(strcmp(tempbuf,""));
} /* http_send() */

/*
 * Send the "read from cddb" command to the server using http
 */
void 
http_read(char *category, unsigned int id)
{
	char tempbuf[84];
	sprintf(tempbuf, "cddb+read+%s+%08x", category, id);
	http_send(tempbuf);
} /* http_read() */

/*
 * The main routine called from the ui
 * returns 0 on success, -1 on err
 */
int 
cddb_request(void)
{
	int i;
	char tempbuf[2000];
	extern int cur_ntracks;
	
	int status, rv  = 0;
	char category[20];
	unsigned int id;
	
	wipe_cdinfo();
	
	switch(cddb.protocol)
	{
	 case 1: /* cddbp */
		Trace((stderr, "USING CDDBP\n"));
		Trace((stderr, "open\n"));
		if (connect_open()) { 
		    Trace((stderr, "connect failed.\n"));
		    return -1;
		}
		connect_getline(tempbuf);
		Trace((stderr, "[%s]\n",tempbuf));
 
		string_makehello(tempbuf,' ');
		Trace((stderr, "%s\n", tempbuf));
		cddbp_send(tempbuf);
		connect_getline(tempbuf);
		Trace((stderr, "[%s]\n",tempbuf));
		
		Trace((stderr, "query\n"));
		sprintf(tempbuf, "cddb query %08x %d",thiscd.cddbid,thiscd.ntracks);
		for (i = 0; i < cur_ntracks; i++)
		  if (thiscd.trk[i].section < 2)
		    sprintf(tempbuf + strlen(tempbuf), " %d",
			    thiscd.trk[i].start);
		sprintf(tempbuf + strlen(tempbuf), " %d\n", thiscd.length);
		Trace((stderr, ">%s<\n",tempbuf));
		cddbp_send(tempbuf);
		connect_getline(tempbuf);
		Trace((stderr, "[%s]\n",tempbuf));
		
		status=atoi(tempbuf);
		Trace((stderr, "status:%d\n",status));
	
		if(status == 200) /* Exact match */
		{
			sscanf(tempbuf,"%d %s %08x",&status,category,&id);
			cddbp_read(category,id);
			connect_read_entry();
		}
		else if(status == 211) /* Unexact match, multiple possible
				   * Hack: always use first. */
		{
			connect_getline(tempbuf);
			Trace((stderr, "[%s]\n",tempbuf));
			sscanf(tempbuf,"%s %08x",category,&id);
			while(strcmp(tempbuf,".")) {
			  connect_getline(tempbuf);
			  Trace((stderr, "ignoring [%s]\n",tempbuf));
			}
			cddbp_read(category,id);
			connect_read_entry();
		}
		else
			rv = -1;
		
		cddbp_send("quit");
		connect_close();
		Trace((stderr, "close\n"));
		return rv;
	 case 2: /* http */
	 case 3: /* http proxy */
		Trace((stderr, "USING HTTP%s\n",
		       (cddb.protocol == 3) ? " WITH PROXY" : ""));
		Trace((stderr, "query\n"));
		sprintf(tempbuf, "cddb+query+%08x+%d",thiscd.cddbid,thiscd.ntracks);
		for (i = 0; i < cur_ntracks; i++)
		  if (thiscd.trk[i].section < 2)
		    sprintf(tempbuf + strlen(tempbuf), "+%d",
			    thiscd.trk[i].start);
		sprintf(tempbuf + strlen(tempbuf), "+%d", thiscd.length);
		Trace((stderr, ">%s<\n",tempbuf));
		if (connect_open()) { 
		    Trace((stderr, "connect failed.\n"));
		    return -1;
		}
		http_send(tempbuf);
		connect_getline(tempbuf);
		Trace((stderr, "[%s]\n",tempbuf));
		
		status=atoi(tempbuf);
		if(status == 200) /* Exact match */
		{
			connect_close();
			connect_open();
			if (connect_open()) { 
			    Trace((stderr, "connect failed.\n"));
			    return -1;
			}
			sscanf(tempbuf,"%d %s %08x",&status,category,&id);
			http_read(category,id);
			connect_read_entry();
		}
		else if(status == 211) /* Unexact match, multiple possible
				   * Hack: always use first. */
		{
			connect_getline(tempbuf);
			sscanf(tempbuf,"%s %08x",category,&id);
			while(strcmp(tempbuf,"."))
			  connect_getline(tempbuf);
			connect_close();
			if (connect_open()) { 
			    Trace((stderr, "connect failed.\n"));
			    return -1;
			}
			http_read(category,id);
			connect_read_entry();
		}
		else
			rv = -1;
		/* moved close above break */
		connect_close();
		return rv;
	 default: /* off */
		break;
	}
	return -1;
} /* cddb_request() */
